"""Lexicographic acceptance and PF/Born tie resolution."""
from __future__ import annotations
from dataclasses import dataclass
from typing import List, Tuple
import numpy as np

@dataclass(frozen=True)
class Candidate:
    x: int
    y: int
    new_dx: int  # lateral move in this step: -1, 0, +1
    last_dx: int  # previous lateral move
    # r1: prefer staying in same column (boolean score, 0 better than 1)
    # r2: forbid immediate zigzag (boolean score, 0 pass, 1 fail)
    # r3: reserved 0
    def residual_tuple(self) -> Tuple[int, int, int]:
        r1 = 0 if self.new_dx == 0 else 1
        zigzag = (self.last_dx == -1 and self.new_dx == +1) or (self.last_dx == +1 and self.new_dx == -1)
        r2 = 0 if not zigzag else 1
        return (r1, r2, 0)

def lex_min(cands: List[Candidate]) -> List[Candidate]:
    scores = [c.residual_tuple() for c in cands]
    best = min(scores)
    return [c for c, s in zip(cands, scores) if s == best]

def tie_kernel_binary(cands: List[Candidate], eta: float = 1e-6) -> np.ndarray:
    n = len(cands)
    A = np.zeros((n, n), dtype=float)
    # Binary adjacency by physical proximity among the tied candidates (|Δx| <= 1)
    for i, ci in enumerate(cands):
        for j, cj in enumerate(cands):
            if i == j:
                continue
            if abs(ci.x - cj.x) <= 1:
                A[i, j] = 1.0
    # Regularize to ensure primitivity
    A = A + eta * np.eye(n, dtype=float)
    return A

def pf_born_sample(A: np.ndarray, rng) -> int:
    # Perron–Frobenius eigenvector of a small positive matrix
    w, V = np.linalg.eig(A)
    idx = np.argmax(w.real)
    v = V[:, idx].real
    v = np.abs(v)
    probs = (v * v) / np.sum(v * v)
    return rng.choice(probs)

def accept_next(cands: List[Candidate], rng, eta: float = 1e-6) -> Candidate:
    if not cands:
        raise ValueError("Empty candidate set")
    tied = lex_min(cands)
    if len(tied) == 1:
        return tied[0]
    A = tie_kernel_binary(tied, eta=eta)
    k = pf_born_sample(A, rng)
    return tied[k]
